#include "stdafx.h"

struct SCPFileHeader {
	uint8_t		mSignature[3];
	uint8_t		mVersion;
	uint8_t		mDiskType;
	uint8_t		mNumRevs;
	uint8_t		mStartTrack;
	uint8_t		mEndTrack;
	uint8_t		mFlags;
	uint8_t		mBitCellEncoding;
	uint8_t		mSides;
	uint8_t		mReserved;
	uint32_t	mChecksum;
	uint32_t	mTrackOffsets[166];
};

static_assert(sizeof(SCPFileHeader) == 0x2A8, "");

struct SCPTrackRevolution {
	uint32_t	mTimeDuration;
	uint32_t	mDataLength;
	uint32_t	mDataOffset;
};

struct SCPTrackHeader {
	uint8_t		mSignature[3];
	uint8_t		mTrackNumber;
};

void scp_read(RawDisk& raw_disk, const char *path, int selected_track) {
	printf("Reading SuperCard Pro image: %s\n", path);

	FILE *fi = fopen(g_inputPath.c_str(), "rb");
	if (!fi)
		fatalf("Unable to open input file: %s.\n", path);

	SCPFileHeader fileHeader = {0};
	if (1 != fread(&fileHeader, sizeof(fileHeader), 1, fi))
		fatal_read();

	if (memcmp(fileHeader.mSignature, "SCP", 3))
		fatalf("Input file does not start with correct SCP image signature.\n");

	const bool tpi96 = (fileHeader.mFlags & 0x02) != 0;
	const bool double_sided = (fileHeader.mSides != 1);
	const int track_step = 2; //(tpi96 ? 2 : 1) * 2;

	if (g_verbosity >= 1)
		printf("SCP: Image is %d tpi, %s-sided\n", tpi96 ? 96 : 48, double_sided ? "double" : "single");

	if (fileHeader.mNumRevs <= 1) {
		printf(
			"Warning: Only one disk revolution found in image. Atari disks are not\n"
			"         index aligned and require at least two revolutions.\n");
	}

	std::vector<uint16_t> data_buf;
	for(int i=0; i<40; ++i) {
		if (selected_track >= 0 && i != selected_track)
			continue;

		const int track = i * track_step;

		const uint32_t track_offset = fileHeader.mTrackOffsets[track];
		if (!track_offset)
			continue;

		if (fseek(fi, track_offset, SEEK_SET))
			fatal_read();

		SCPTrackHeader track_hdr = {};
		if (1 != fread(&track_hdr, sizeof(track_hdr), 1, fi))
			fatal_read();

		if (memcmp(track_hdr.mSignature, "TRK", 3))
			fatalf("SCP raw track %d has broken header at %08x with incorrect signature.", track, track_offset);

		std::vector<SCPTrackRevolution> revs(fileHeader.mNumRevs);
		if (1 != fread(revs.data(), sizeof(SCPTrackRevolution)*fileHeader.mNumRevs, 1, fi))
			fatal_read();

		// initialize raw track parameters
		RawTrack& raw_track = raw_disk.mTracks[i];
		raw_track.mIndexTimes.push_back(0);

		// compute average revolution time
		uint32_t total_rev_time = 0;
		for(auto it = revs.begin(), itEnd = revs.end(); it != itEnd; ++it) {
			const auto& rev = *it;

			total_rev_time += rev.mTimeDuration;
			raw_track.mIndexTimes.push_back(total_rev_time);
		};

		raw_track.mSamplesPerRev = (float)total_rev_time / (float)fileHeader.mNumRevs;
		raw_track.mTrack = i;
		raw_track.mSpliceStart = -1;
		raw_track.mSpliceEnd = -1;

		if (g_verbosity >= 1)
			printf("Track %d: %.2f RPM\n", i, 60.0 / (raw_track.mSamplesPerRev * 0.000000025));

		// parse out flux transitions for each rev
		uint32_t time = 0;

		for(auto it = revs.begin(), itEnd = revs.end(); it != itEnd; ++it) {
			const auto& rev = *it;

			if (rev.mDataLength > 0x1000000)
				fatalf("SCP raw track %u at %08X has an excessively long sample list.\n", track, track_offset);

			data_buf.resize(rev.mDataLength);

			if (fseek(fi, track_offset + rev.mDataOffset, SEEK_SET) < 0
				|| 1 != fread(data_buf.data(), rev.mDataLength * 2, 1, fi))
				fatal_read();

			for(auto it2 = data_buf.begin(), it2End = data_buf.end(); it2 != it2End; ++it2) {
				const auto& offset_be = *it2;

				// need to byte-swap
				uint32_t offset = ((offset_be << 8) & 0xff00) + ((offset_be >> 8) & 0xff);

				if (offset) {
					time += offset;
					raw_track.mTransitions.push_back(time);
				} else
					time += 0x10000;
			};
		};
	}

	fclose(fi);
};

void scp_write(const RawDisk& raw_disk, const char *path, int selected_track) {
	int maxrevs = 5;

	// go through all tracks we'll be writing, and find out the max number of revs we have
	for(int i=0; i<40; ++i) {
		if (selected_track >= 0 && selected_track != i)
			continue;

		int revs = (int)raw_disk.mTracks[i].mIndexTimes.size() - 1;

		if (revs > 0 && revs < maxrevs)
			maxrevs = revs;
	}
	
	printf("Writing SuperCard Pro image with %u revolutions: %s\n", maxrevs, path);

	FILE *fo = fopen(path, "wb");
	if (!fo)
		fatalf("Cannot open %s for write\n", path);

	SCPFileHeader filehdr = {0};

	filehdr.mSignature[0] = 'S';
	filehdr.mSignature[1] = 'C';
	filehdr.mSignature[2] = 'P';
	filehdr.mVersion = 0x15;		// We aren't SCP, so we just use ver 1.5
	filehdr.mDiskType = 0x10;		// Atari 800
	filehdr.mNumRevs = maxrevs;
	filehdr.mStartTrack = 0;
	filehdr.mEndTrack = 79;
	filehdr.mFlags = 0x07;
	filehdr.mBitCellEncoding = 0;
	filehdr.mSides = 0;
	filehdr.mChecksum = 0;

	// write out initial header
	fwrite(&filehdr, sizeof filehdr, 1, fo);

	for(int i=0; i<40; ++i) {
		if (selected_track >= 0 && selected_track != i)
			continue;

		const auto& track_info = raw_disk.mTracks[i];

		// skip track if it is empty
		if (track_info.mIndexTimes.size() < (size_t)maxrevs + 1)
			continue;

		// rescale all transitions and index times
		// we need 25ns samples and a 360 RPM rotational speed
		const double sample_scale = (40000000.0 / 6.0) / (double)track_info.mSamplesPerRev;

		std::vector<uint32_t> new_samples;
		const auto& transitions = track_info.mTransitions;
		
		new_samples.reserve(transitions.size());
		for(std::vector<uint32_t>::const_iterator it = std::lower_bound(transitions.begin(), transitions.end(), track_info.mIndexTimes.front()),
			itEnd = std::upper_bound(it, transitions.end(), track_info.mIndexTimes[maxrevs]); it != itEnd; ++it) {
			new_samples.push_back((uint32_t)((double)*it * sample_scale + 0.5));
		}

		std::vector<uint32_t> new_index_marks;
		for(auto it = track_info.mIndexTimes.begin(), itEnd = track_info.mIndexTimes.end(); it != itEnd; ++it) {
			new_index_marks.push_back((uint32_t)((double)*it * sample_scale + 0.5));
		}

		// encode all samples
		std::vector<uint8_t> bitdata;
		std::vector<uint32_t> new_sample_offsets;

		if (new_samples.size() > 1) {
			uint32_t error = 0;
			uint32_t last_time = new_index_marks.front();

			for(auto it = new_samples.begin(), itEnd = new_samples.end(); it != itEnd; ++it) {
				uint32_t delay = *it - last_time;
				last_time = *it;

				// we can't encode a sample of 0, so we need to push such transitions
				if (delay < error + 1) {		// delay - error < 1
					error += 1 - delay;
					delay = 1;
				}

				// encode
				new_sample_offsets.push_back((uint32_t)bitdata.size());

				while(delay >= 0x10000) {
					bitdata.push_back(0);
					bitdata.push_back(0);
					delay -= 0x10000;
				}

				bitdata.push_back((uint8_t)(delay >> 8));
				bitdata.push_back((uint8_t)(delay >> 0));
			}
		}

		if (bitdata.size() & 2) {
			bitdata.push_back(0);
			bitdata.push_back(0);
		}

		new_sample_offsets.push_back((uint32_t)bitdata.size());

		uint32_t track_data[16] = {0};

		memcpy(&track_data[0], "TRK", 3);
		((uint8_t *)track_data)[3] = i;

		for(int j=0; j<maxrevs; ++j) {
			// index time
			track_data[3*j + 1] = new_index_marks[j + 1] - new_index_marks[j];

			// track data offset
			auto sample_start = std::lower_bound(new_samples.begin(), new_samples.end(), new_index_marks[j]);
			auto sample_end = std::lower_bound(new_samples.begin(), new_samples.end(), new_index_marks[j+1]);
			uint32_t data_start = new_sample_offsets[sample_start - new_samples.begin()];
			uint32_t data_end = new_sample_offsets[sample_end - new_samples.begin()];

			track_data[3*j + 3] = sizeof(track_data) + data_start;

			// track length (in encoded values)
			track_data[3*j + 2] = (data_end - data_start) >> 1;
		}

		// update checksum
		filehdr.mChecksum += ComputeByteSum(track_data, sizeof track_data);
		filehdr.mChecksum += ComputeByteSum(bitdata.data(), bitdata.size());

		// set track offset and update checksum
		uint32_t track_offset = (uint32_t)ftell(fo);
		filehdr.mTrackOffsets[2*i] = track_offset;

		// write it out
		fwrite(track_data, sizeof track_data, 1, fo);
		fwrite(bitdata.data(), bitdata.size(), 1, fo);
	}

	// rewrite file header
	filehdr.mChecksum += ComputeByteSum(filehdr.mTrackOffsets, sizeof filehdr.mTrackOffsets);

	fseek(fo, 0, SEEK_SET);
	fwrite(&filehdr, sizeof filehdr, 1, fo);
	fclose(fo);
}
